<?php

namespace App\Http\Controllers\User;

use App\Models\Gateway;
use App\Constants\Status;
use Illuminate\Http\Request;
use App\Models\GatewayCurrency;
use App\Models\AdminNotification;
use App\Http\Controllers\Controller;
use App\Models\UserWallet;

class WalletController extends Controller
{
    public function index(){

        $pageTitle = "Fund Wallet";
        $user = auth()->user();

        $paystackDetails = null;
        $isPaystackGatewayEnabled = $this->isPaystackGatewayEnabled();

        if ($this->isPaystackGatewayEnabled()) {

            $paystack = Gateway::where('name', 'Paystack')->first();
            $gatewayParameter = json_decode($paystack->gateway_parameters);

            $paystackDetails = [
                'gatewayParameter' => $gatewayParameter,
                'user' => auth()->user()
            ];
        }

        return view('Template::user.fund_wallet', compact('pageTitle','user', 'paystackDetails', 'isPaystackGatewayEnabled'));

    }

    public function paystackCallback(Request $request)
    {

        if ($this->isPaystackGatewayEnabled()) {


        $paystack = Gateway::where('name', 'Paystack')->first();
        $gatewayParameter = json_decode($paystack->gateway_parameters);
        $secret_key = $gatewayParameter->secret_key->value;

        $reference = $request->query('reference');

        $result = [];
        //The parameter after verify/ is the transaction reference to be verified
        $url = 'https://api.paystack.co/transaction/verify/' . $reference;
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Bearer ' . $secret_key]);
        $response = curl_exec($ch);
        curl_close($ch);

        if ($response) {
            $result = json_decode($response, true);

            if ($result) {
                if ($result['data']) {


                    if ($result['data']['status'] == 'success') {

                        $paidAmount = $result['data']['amount'] / 100; // Convert amount to actual value

                        $user = auth()->user();

                        $payment = UserWallet::create([
                            'user_id' => $user->id,
                            'reference' => $reference,
                            'amount' => $paidAmount,
                            'is_manual' => Status::NO,
                            'gateway_id' => $paystack->id,
                            'status' => Status::PAYMENT_SUCCESS,
                        ]);

                        $payment->user->update([
                           'balance' => $paidAmount + $user->balance
                        ]);


                        $notify[] = ['success', 'Payment completed'];

                        return to_route('user.home')->withNotify($notify);


                     }

                    } else {
                        $notify[] = ['error', $result['data']['gateway_response']];
                    }
                }

            } else {
                $notify[] = ['error', 'Something went wrong while executing'];
            }

        }else{
            $notify[] = ['error', "You can't fund your wallet at the moment"];
            return back()->withNotify($notify);

        }

    }

    protected function isPaystackGatewayEnabled()
    {
        $paystackGatewayEnabled = Gateway::where('name', 'Paystack')
                                        ->where('status', Status::ENABLE)
                                        ->exists();

        return $paystackGatewayEnabled;
    }

    public function manual(Request $request){

        $pageTitle = 'Manual Funding';

        $amount = $request->query('am');
        $method = $request->query('mt');

        $gate = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->where('method_code', $method)->first();
        if (!$gate) {
            $notify[] = ['error', 'Invalid gateway'];
            return back()->withNotify($notify);
        }

        if ($gate->min_amount > $amount || $gate->max_amount < $amount) {
            $notify[] = ['error', 'Please follow deposit limit'];
            return back()->withNotify($notify);
        }

        $charge = $gate->fixed_charge + ($amount * $gate->percent_charge / 100);
        $payable = $amount + $charge;
        $final_amo = $payable * $gate->rate;

        $data = [
            'gateway' => $gate,
            'charge' => $charge,
            'payable' => $payable,
            'amount' => $amount,
            'final_amo' => $final_amo,
            'method_currency' => $gate->currency
        ];


        return view('Template::user.manual_wallet_fund', compact('pageTitle', 'data'));

    }

    public function manualFund(Request $request)
    {

        $request->validate([
            'payment_receipt' => ['required', 'mimes:jpg,png,jpeg,pdf'],
            'amount' => ['required']
        ]);

        $reference = uniqid('M');
        $user = auth()->user();


        if($request->hasFile('payment_receipt')){

           $receiptFileName = fileUploader($request->payment_receipt, 'assets/payment_receipt');
        }

        if($receiptFileName && $receiptFileName !== null){

            $fundWallet = UserWallet::create([
                'user_id' => auth()->id(),
                'reference' =>$reference,
                'amount'=> $request->amount,
                'status'=> Status::PAYMENT_PENDING,
                'is_manual' => Status::YES,
                'payment_receipt' => $receiptFileName
            ]);
        }

        if($fundWallet){

            $adminNotification = new AdminNotification();
            $adminNotification->user_id = $user->id;
            $adminNotification->title = 'Balance funding request from '.$user->username;
            $adminNotification->click_url = urlPath('admin.deposit.details',$user->id);
            $adminNotification->save();

            $notify[] = ['success', 'Your payment request has been taken'];
            return to_route('user.home')->withNotify($notify);
        }else {
            $notify[] = ['error', 'There is an error with the submission process'];
            return to_route('user.dashboard')->withNotify($notify);
        }
    }


}
